<?php
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

class WPPB_Two_Factor_Authenticator {

    public function __construct( ) {

        add_action( 'admin_init', array( $this, 'wppb_two_factor_authentication_settings_update' ) );

        // Clear edit profile pages cache when pages are saved
        add_action( 'save_post', array( $this, 'clear_edit_profile_pages_cache' ) );

        if ( ! class_exists('WPPB_Base32') ) {
            require_once( WPPB_PAID_PLUGIN_DIR.'/features/two-factor-authentication/assets/lib/class-WPPBBase32.php' );
        }

	    $wppb_two_factor_authentication_settings = get_option( 'wppb_two_factor_authentication_settings', 'not_found' );

	    $enabled = 'no';
	    if ( !empty( $wppb_two_factor_authentication_settings['enabled'] ) ) {
		    $enabled = $wppb_two_factor_authentication_settings['enabled'];
	    }

        if ( $enabled === 'yes' ) {

	        add_filter( 'wp_authenticate_user', array( $this, 'login_error_message_handler' ), 10, 2 );
	        add_filter( 'wp_login_errors', array( $this, 'back_end_errors_filter' ), 10, 2 );

	        // Hook into user registration and role changes for forced 2FA
	        add_action( 'user_register', array( $this, 'handle_new_user_registration' ) );
	        add_action( 'set_user_role', array( $this, 'handle_user_role_change' ), 10, 3 );

	        // Hook into login redirect for forced 2FA enforcement
	        add_filter( 'login_redirect', array( $this, 'handle_forced_2fa_login_redirect' ), 10, 3 );

	        // Hook into admin notices for forced 2FA warnings
	        add_action( 'admin_notices', array( $this, 'show_forced_2fa_admin_notices' ) );

	        // Hook into template redirect for already logged-in users
	        add_action( 'template_redirect', array( $this, 'handle_already_logged_in_users' ) );
	        add_action( 'admin_init', array( $this, 'handle_already_logged_in_users' ) );

            // Bypass Custom Redirects if 2FA is required
            add_filter( 'wppb_custom_redirect_url_filter', array( $this, 'bypass_custom_redirects_for_2fa' ), 10, 4 );

	        if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
		        add_action( 'wp_ajax_WPPBAuth_new_secret', array( $this, 'ajax_new_secret' ) );
		        add_action( 'wp_ajax_WPPBAuth_check_code', array( $this, 'ajax_check_code' ) );
		        add_action( 'wp_ajax_WPPBAuth_regenerate_backup_codes', array( $this, 'ajax_regenerate_backup_codes' ) );
		        add_action( 'wp_ajax_nopriv_WPPBAuth_field_on_login_form', array( $this, 'ajax_add_auth_field_to_login_form' ) );
	        }

	        add_action( 'show_user_profile', array( $this, 'add_field_to_backend_edit_profile_form' ) );
	        add_action( 'edit_user_profile', array( $this, 'add_field_to_backend_edit_profile_form' ) );
	        add_action( 'wppb_backend_save_form_field', array( $this, 'handle_backend_edit_profile_update' ), 10, 4 );

	        add_filter( 'wppb_filter_form_args_before_output', array(
		        $this,
		        'add_field_to_frontend_edit_profile_form'
	        ) );
	        add_filter( 'wppb_output_form_field_two-factor-authentication', array(
		        $this,
		        'frontend_edit_profile_field'
	        ), 10, 6 );
	        add_action( 'wppb_after_saving_form_values', array( $this, 'handle_frontend_edit_profile_update' ), 10, 2 );

	        add_filter( 'wppb_form_fields', array( $this, 'add_field_to_pb_field_validation_sequence' ), 10, 2 );
	        add_filter( 'wppb_check_form_field_two-factor-authentication', array(
		        $this,
		        'form_field_validation'
	        ), 10, 4 );

	        add_action( 'login_footer', array( $this, 'add_field_to_backend_login_form' ) );
	        add_action( 'login_form_bottom', array( $this, 'add_field_to_frontend_login_form' ), 10, 2 );

            add_filter( 'wppb_login_form_args', array( $this, 'add_2fa_class_to_login_form' ), 10, 1 );
        }
    }

    /**
     * Handle 2FA scripts
     */
    function enqueue_2fa_scripts( ) {
        wp_enqueue_script( 'jquery' );
        wp_enqueue_script( 'wppb_qrcode_script', plugin_dir_url( __FILE__ ) .'assets/js/jquery.qrcode.min.js', array( 'jquery' ), PROFILE_BUILDER_VERSION, true);
        wp_enqueue_script( 'wppb_2fa_script', plugin_dir_url( __FILE__ ) .'assets/js/wppb-2fa.js', array( 'jquery', 'wppb_qrcode_script' ), PROFILE_BUILDER_VERSION, true );
        wp_localize_script( 'wppb_2fa_script', 'wppb_2fa_script_vars', array(
                'WPPBAuthNonce'        => wp_create_nonce( 'WPPBAuthaction' ),
                'ajaxurl'              => admin_url( 'admin-ajax.php' ),
                'valid'                => __('Valid', 'profile-builder' ),
                'invalid'              => __('Invalid', 'profile-builder' ),
                'backupCodes'          => __('Backup Codes', 'profile-builder' ),
                'regenerateBackupCodes'=> __('Regenerate Backup Codes', 'profile-builder' ),
                'copyCodes'            => __('Copy Codes', 'profile-builder' ),
                'downloadCodes'        => __('Download Codes', 'profile-builder' ),
                'regenerateConfirm'    => __('Regenerating will invalidate all existing codes. Continue?', 'profile-builder' ),
                'eachCodeOnce'         => __('Each code can only be used once. Please save these codes in a secure location.', 'profile-builder' ),
                'remainingCodes'       => __('remaining codes', 'profile-builder' ),
            )
        );
    }

    /**
     * Add 2FA settings field to the frontend Edit Profile form
     */
    function add_field_to_frontend_edit_profile_form($args ) {
        $wppb_two_factor_authentication_settings = get_option( 'wppb_two_factor_authentication_settings', 'not_found' );
        if( isset( $wppb_two_factor_authentication_settings['enabled'] ) && $wppb_two_factor_authentication_settings['enabled'] === 'yes' ) {
            // Check if we should hide other fields when redirected for 2FA setup
            $hide_other_fields = false;
            if ( isset( $_GET['wppb_2fa_required'] ) && $_GET['wppb_2fa_required'] === '1' ) {
                if ( apply_filters( 'wppb_2fa_hide_other_fields_on_redirect', true ) ) {
                    $hide_other_fields = true;
                }
            }

            if ( $hide_other_fields ) {
                // Only show the 2FA field
                $args['form_fields'] = $this->add_field_info( array() );
            } else {
                // Add 2FA field to existing fields
                $args['form_fields'] = $this->add_field_info($args['form_fields']);
            }
        }
        return $args;
    }

    function add_field_info( $array ){
        $array[] = array(
            'field-title' => __( 'Two-Factor Authentication', 'profile-builder' ),
            'field' => 'Two-Factor Authentication',
            'meta-name' => '2fa_settings',
            'id' => '2fa_settings',
        );
        return $array;
    }

    /**
     * Add 2FA settings field to frontend profile page
     */

    function frontend_edit_profile_field($output, $form_location, $field, $user_id ){
        if ( $field['field'] === 'Two-Factor Authentication' ){
            if ( $form_location === 'edit_profile' ){

                $wppb_two_factor_authentication_settings = get_option( 'wppb_two_factor_authentication_settings', 'not_found' );
                if( isset( $wppb_two_factor_authentication_settings['enabled'], $wppb_two_factor_authentication_settings['roles'] ) && $wppb_two_factor_authentication_settings['enabled'] === 'yes' ) {

                    if( ( !is_multisite() && current_user_can( 'edit_users' ) ) || ( is_multisite() && ( current_user_can( 'remove_users' ) || current_user_can( 'manage_options' ) ) ) ){
                        if ( isset( $_REQUEST['edit_user'] ) ) {
                            $user_id = absint( $_REQUEST['edit_user'] );
                        }
                    } else {
                        $user = wp_get_current_user();
                        $user_id = $user->ID;
                    }

                    $user_meta = get_userdata( $user_id );

                    if ( is_super_admin( $user_id ) ) {
                        $user_meta->roles[] = 'administrator';
                    }

                    if ( $this->should_user_see_field( $wppb_two_factor_authentication_settings['roles'], $user_meta ) ) {

                        $secret      = trim( get_user_option( 'wppb_auth_secret', $user_id ) );
                        $enabled     = trim( get_user_option( 'wppb_auth_enabled', $user_id ) );
                        $relaxedmode = trim( get_user_option( 'wppb_auth_relaxedmode', $user_id ) );
                        $description = trim( get_user_option( 'wppb_auth_description', $user_id ) );

                        // Check if 2FA is forced for this user
                        $is_forced = $this->is_2fa_forced_for_user( $user_id );
                        $grace_period_expired = $this->is_grace_period_expired( $user_id );
                        $remaining_days = $this->get_remaining_grace_period_days( $user_id );

                        // In case the user has no secret ready ( new install ), we create one. or use the one they just posted
                        if ( '' === $secret ) {
                            $secret = array_key_exists( 'wppb_auth_secret', $_REQUEST ) ? sanitize_text_field( $_REQUEST['wppb_auth_secret'] ) : $this->create_secret( );
                        }

                        if ( '' === $description ) {
                            if ( is_multisite( ) && ( 1 < count( get_blogs_of_user( $user_id ) ) || is_super_admin( ) ) ) {
                                $description = get_blog_details( get_network( )->id )->blogname;
                            } else {
                                $description = get_bloginfo( 'name' );
                            }
                        }

                        $this->enqueue_2fa_scripts( );

                        // Show forced 2FA notice if applicable
                        $notice_html = '';
                        if ( $is_forced && $enabled !== 'enabled' ) {
                            $notice_class = 'wppb-error';
                            if ( !$grace_period_expired && $remaining_days > 0 ) {
                                $notice_class = 'wppb-warning';
                                $notice_message = sprintf(
                                    __( 'You must set up Two-Factor Authentication within %d days. Your role requires 2FA for security.', 'profile-builder' ),
                                    $remaining_days
                                );
                            } else {
                                $notice_message = __( 'Two-Factor Authentication is required for your role. Please set it up now.', 'profile-builder' );
                            }

                            $notice_html = '<div class="wppb-form-field wppb-2fa-forced-notice ' . esc_attr( $notice_class ) . '">
                                <p><strong>' . esc_html( $notice_message ) . '</strong></p>
                            </div>';
                        }

                        $output .= '
                            <ul class="wppb-2fa-fields">
                                <li class="wppb-form-field wppb_2fa_heading"><h3>' . esc_html__( 'Two-Factor Authentication', 'profile-builder' ) . '</h3></li>
                                ' . $notice_html . '
                                <li class="wppb-form-field wppb_auth_enabled">
                                    <input name="wppb_auth_enabled" id="wppb_auth_enabled" type="checkbox" ' . checked( $enabled, 'enabled' ) . '/>
                                    <label for="wppb_auth_enabled" style="display: inline-block; width: auto;">' . esc_html__( 'Activate', 'profile-builder' ) . '</label>
                                </li>
                                <div id="wppb_auth_active">
                                    <li class="wppb-form-field wppb_auth_relaxedmode">
                                        <input name="wppb_auth_relaxedmode" id="wppb_auth_relaxedmode" type="checkbox" ' . checked( $relaxedmode, 'enabled' ) . '/>
                                        <label for="wppb_auth_relaxedmode" style="display: inline-block; width: auto;">' . esc_html__( 'Relaxed Mode', 'profile-builder' ) . '</label>
                                        <span class="wppb-description-delimiter">' . esc_html__( "Allow for more time drift on your phone clock ( &#177;4 min ).", "profile-builder" ) . '</span>
                                    </li>
                                    <li class="wppb-form-field wppb_auth_description'. apply_filters( 'wppb_2fa_field_extra_css_class', '', 'wppb_auth_description') .'">
                                        <label for="wppb_auth_description">' . esc_html__( 'Description', 'profile-builder' ) . '</label>
                                        <input name="wppb_auth_description" id="wppb_auth_description" type="text" value="' . esc_attr( $description ) . '"/>
                                        <span class="wppb-description-delimiter">' . esc_html__( 'Description that you\'ll see in the Authenticator app.', 'profile-builder' ) . '</span>
                                    </li>
                                    <li class="wppb-form-field wppb_auth_secret'. apply_filters( 'wppb_2fa_field_extra_css_class', '', 'wppb_auth_secret') .'">
                                        <label for="wppb_auth_secret">' . esc_html__( 'Secret', 'profile-builder' ) . '</label>
                                        <input name="wppb_auth_secret" id="wppb_auth_secret" type="text" readonly="readonly" size="25" value="' . esc_attr( $secret ) . '"/>
                                    </li>
                                    <li id="wppb_auth_secret_buttons" style="">
                                        <input name="wppb_auth_newsecret" id="wppb_auth_newsecret" value="' . esc_html__( 'New Secret', 'profile-builder' ) . '" type="button" class="button wppb_auth_button wppb_auth_new_button" />
                                        <input name="wppb_show_qr" id="wppb_show_qr" value="' . esc_html__( 'QR Code', 'profile-builder' ) . '" type="button" class="button wppb_auth_button wppb_auth_qr_button" onclick="ShowOrHideQRCode( )" />
                                    </li>
                                    <li id="wppb_auth_QR_INFO" style="display: none">
                                        <span class="wppb-description-delimiter">' . esc_html__( 'Scan this with the Authenticator app:', 'profile-builder' ) . '</span>
                                        <div id="wppb_auth_QRCODE"></div>
                                    </li>
                                    <li class="wppb-form-field wppb_auth_verify'. apply_filters( 'wppb_2fa_field_extra_css_class', '', 'wppb_auth_passw') .'">
                                        <label for="wppb_auth_passw">' . esc_html__( 'Verify TOTP', 'profile-builder' ) . '</label>
                                        <input name="wppb_auth_passw" id="wppb_auth_passw" type="text"/>
                                    </li>
                                    <li id="wppb_auth_verify_buttons" style="">
                                        <input name="wppb_auth_verify_button" id="wppb_auth_verify_button" value="' . esc_html__( 'Check', 'profile-builder' ) . '" type="button" class="button wppb_auth_button wppb_auth_verify_button" />
                                        <input name="wppb_auth_verify_indicator" id="wppb_auth_verify_indicator" value="" type="button" class="button wppb_auth_button wppb_auth_verify_indicator" disabled />
                                        <input type="hidden" value="" name="wppb_auth_verify_result" id="wppb_auth_verify_result"/>
                                    </li>
                                    ' . $this->get_backup_codes_frontend_html( $user_id, $enabled ) . '
                                </div>
                            </ul>
                            ';
                    }
                }
            }
        }
        return $output;
    }

    /**
     * Get backup codes HTML for frontend profile form
     *
     * @param int $user_id User ID
     * @param string $enabled 2FA enabled status
     * @return string HTML output
     */
    function get_backup_codes_frontend_html( $user_id, $enabled ) {
        $remaining_codes = $this->get_remaining_backup_codes_count( $user_id );

        $output = '';

        // Show backup codes management
        if ( $enabled === 'enabled' ) {
            $output .= '
                <li class="wppb-form-field wppb_auth_backup'. apply_filters( 'wppb_2fa_field_extra_css_class', '', 'wppb_auth_backup') .'">
                    <label for="wppb_auth_backup">' . esc_html__( 'Backup Codes', 'profile-builder' ) . '</label>
                </li>
                <li id="wppb_auth_backup_buttons" style="">
                    <input name="wppb_auth_backup_button" id="wppb_auth_backup_button" value="' . esc_html__( 'Regenerate Backup Codes', 'profile-builder' ) . '" type="button" class="button wppb_auth_button wppb_auth_backup_button" />';

            if ( $remaining_codes <= 3 && $remaining_codes > 0 ) {
                $output .= '
                    <input name="wppb_auth_backup_indicator" id="wppb_auth_backup_indicator" value="' . esc_html( sprintf( __( '%d remaining codes!', 'profile-builder' ), $remaining_codes ) ) . '" type="button" class="button wppb_auth_button wppb_auth_backup_indicator wppb-backup-codes-count low" disabled />';
            } elseif ( $remaining_codes == 0 ) {
                $output .= '
                    <input name="wppb_auth_backup_indicator" id="wppb_auth_backup_indicator" value="' . esc_html( sprintf( __( 'No remaining codes!', 'profile-builder' ), $remaining_codes ) ) . '" type="button" class="button wppb_auth_button wppb_auth_backup_indicator wppb-backup-codes-count zero" disabled />';
            } else {
                $output .= '
                    <input name="wppb_auth_backup_indicator" id="wppb_auth_backup_indicator" value="' . esc_html( sprintf( __( '%d remaining codes', 'profile-builder' ), $remaining_codes ) ) . '" type="button" class="button wppb_auth_button wppb_auth_backup_indicator wppb-backup-codes-count" disabled />';
            }

            $output .= '
                </li>';
        }

        return $output;
    }

    function handle_frontend_edit_profile_update($global_request, $args ) {
        if( $args['form_type'] === 'edit_profile' ) {

            if ( ( ( !is_multisite() && current_user_can( 'edit_users' ) ) || ( is_multisite() && ( current_user_can( 'remove_users' ) || current_user_can( 'manage_options' ) ) ) ) && isset( $global_request['edit_user'] ) ){
                $user_id = $global_request['edit_user'];
            } else {
                $user    = wp_get_current_user();
                $user_id = $user->ID;
            }

            $was_2fa_enabled = get_user_option( 'wppb_auth_enabled', $user_id );
            $is_now_enabled = isset( $global_request['wppb_auth_enabled'] ) ? 'enabled' : 'disabled';

            update_user_option( $user_id, 'wppb_auth_enabled', $is_now_enabled, true );
            update_user_option( $user_id, 'wppb_auth_relaxedmode', isset( $global_request['wppb_auth_relaxedmode'] ) ? 'enabled' : 'disabled', true );

	        if ( isset( $global_request['wppb_auth_description'] ) )
                update_user_option( $user_id, 'wppb_auth_description', trim( sanitize_text_field( $global_request['wppb_auth_description'] ) ), true );

	        if ( isset( $global_request['wppb_auth_secret'] ) )
                update_user_option( $user_id, 'wppb_auth_secret', trim( $global_request['wppb_auth_secret'] ), true );

            // Generate backup codes when 2FA is enabled for the first time
            if ( $was_2fa_enabled !== 'enabled' && $is_now_enabled === 'enabled' ) {
                $existing_codes = get_user_option( 'wppb_auth_backup_codes', $user_id );
                if ( !$existing_codes || empty( $existing_codes ) ) {
                    $codes = $this->generate_backup_codes( 10 );
                    $this->save_backup_codes( $user_id, $codes['hashed'] );
                }
            }

            // If user just enabled 2FA and was redirected here, redirect them back to intended destination
            if ( $was_2fa_enabled !== 'enabled' && $is_now_enabled === 'enabled' && isset( $_GET['wppb_2fa_required'] ) ) {
                $redirect_url = get_transient( 'wppb_2fa_redirect_' . $user_id );
                if ( $redirect_url ) {
                    delete_transient( 'wppb_2fa_redirect_' . $user_id );
                    wp_redirect( $redirect_url );
                    exit;
                }
            }

        }
    }

    /**
     * Add 2FA settings field to backend profile page
     */
    function add_field_to_backend_edit_profile_form( $user ) {
        $user_id = $user->ID;
        $wppb_two_factor_authentication_settings = get_option( 'wppb_two_factor_authentication_settings', 'not_found' );
        if( ( isset( $wppb_two_factor_authentication_settings['enabled'], $wppb_two_factor_authentication_settings['roles'] ) && $wppb_two_factor_authentication_settings['enabled'] === 'yes' ) ) {
            $user_meta = get_userdata( $user_id );
            if ( is_super_admin( $user_id ) ) {
                $user_meta->roles[] = 'administrator';
            }

            if ( $this->should_user_see_field( $wppb_two_factor_authentication_settings['roles'], $user_meta ) ) {
                $secret		     	= trim( get_user_option( 'wppb_auth_secret', $user_id ) );
                $enabled			= trim( get_user_option( 'wppb_auth_enabled', $user_id ) );
                $relaxedmode		= trim( get_user_option( 'wppb_auth_relaxedmode', $user_id ) );
                $description		= trim( get_user_option( 'wppb_auth_description', $user_id ) );

                // Check if 2FA is forced for this user
                $is_forced = $this->is_2fa_forced_for_user( $user_id );
                $grace_period_expired = $this->is_grace_period_expired( $user_id );
                $remaining_days = $this->get_remaining_grace_period_days( $user_id );

                // In case the user has no secret ready ( new install ), we create one. or use the one they just posted
                if ( '' === $secret ) {
                    $secret = array_key_exists( 'wppb_auth_secret', $_REQUEST ) ? sanitize_text_field( $_REQUEST[ 'wppb_auth_secret' ] ) : $this->create_secret( );
                }

                if ( '' === $description ) {
                    if ( is_multisite( ) && ( 1 < count( get_blogs_of_user( $user_id ) )  || is_super_admin( ) ) ) {
                        $description = sanitize_text_field( get_blog_details( get_network( )->id )->blogname );
                    } else {
                        $description = sanitize_text_field( get_bloginfo( 'name' ) );
                    }
                }

                $this->enqueue_2fa_scripts( );
                wp_enqueue_style( 'wppb-back-end-style', WPPB_PLUGIN_URL . 'assets/css/style-back-end.css', false, PROFILE_BUILDER_VERSION );

                ?>
                    <h3><?php echo esc_html__( 'Two-Factor Authentication Settings', 'profile-builder' ); ?></h3>

                    <?php
                    // Show forced 2FA notice if applicable
                    if ( $is_forced && $enabled !== 'enabled' ) {
                        $notice_class = 'notice notice-error';
                        if ( !$grace_period_expired && $remaining_days > 0 ) {
                            $notice_class = 'notice notice-warning';
                            $notice_message = sprintf(
                                __( 'You must set up Two-Factor Authentication within %d days. Your role requires 2FA for security.', 'profile-builder' ),
                                $remaining_days
                            );
                        } else {
                            $notice_message = __( 'Two-Factor Authentication is required for your role. Please set it up now.', 'profile-builder' );
                        }

                        echo '<div class="' . esc_attr( $notice_class ) . ' is-dismissible">';
                        echo '<p><strong>' . esc_html( $notice_message ) . '</strong></p>';
                        echo '</div>';
                    }
                    ?>
                    <table class="form-table">
                        <tbody>
                            <tr>
                                <th scope="row"><?php echo esc_html__( 'Activate', 'profile-builder' ); ?></th>
                                <td>
                                    <input name="wppb_auth_enabled" id="wppb_auth_enabled" class="tog" type="checkbox"<?php echo checked( $enabled, 'enabled', false ); ?>/>
                                </td>
                            </tr>
                            <tr class="wppb_auth_active wppb_auth_relaxedmode">
                                <th scope="row"><?php echo esc_html__( "Relaxed Mode", "profile-builder" ); ?></th>
                                <td>
                                    <input name="wppb_auth_relaxedmode" id="wppb_auth_relaxedmode" class="tog" type="checkbox"<?php echo checked( $relaxedmode, "enabled", false ); ?>/>
                                    <span class="description"><?php echo esc_html__( "Allow for more time drift on your phone clock ( &#177;4 min ).", "profile-builder" ); ?></span>
                                </td>
                            </tr>
                            <tr class="wppb_auth_active wppb_auth_description">
                                <th><label for="wppb_auth_description"><?php echo esc_html__( 'Description', 'profile-builder' ); ?></label></th>
                                <td>
                                    <input name="wppb_auth_description" id="wppb_auth_description" value="<?php echo esc_html( $description ); ?>"  type="text" size="25" />
                                    <span class="description"><?php echo esc_html__( 'Description that you\'ll see in the Authenticator app on your phone.', 'profile-builder' ); ?></span><br/>
                                </td>
                            </tr>
                            <tr class="wppb_auth_active wppb_auth_secret">
                                <th><label for="wppb_auth_secret"><?php echo esc_html__( 'Secret', 'profile-builder' ); ?></label></th>
                                <td>
                                    <input name="wppb_auth_secret" id="wppb_auth_secret" value="<?php echo esc_attr( $secret ); ?>" readonly="readonly"  type="text" size="25" />
                                    <input name="wppb_auth_newsecret" id="wppb_auth_newsecret" value="<?php echo esc_html__( 'Create new secret', 'profile-builder' ); ?>" type="button" class="button" />
                                    <input name="wppb_show_qr" id="wppb_show_qr" value="<?php echo esc_html__( 'Show/Hide QR code', 'profile-builder' ); ?>" type="button" class="button" onclick="ShowOrHideQRCode( )" />
                                </td>
                            </tr>
                            <tr id="wppb_auth_QR_INFO" style="display: none">
                                <th></th>
                                <td>
                                    <span class="description"><br/><?php echo esc_html__( 'Scan this with the Authenticator app:', 'profile-builder' ); ?></span>
                                    <br/>
                                    <div id="wppb_auth_QRCODE"></div>
                                </td>
                            </tr>
                            <tr class="wppb_auth_active wppb_auth_verify">
                                <th><label for="wppb_auth_passw"><?php echo esc_html__( 'Verify TOTP', 'profile-builder' ); ?></label></th>
                                <td>
                                    <input name="wppb_auth_passw" id="wppb_auth_passw" type="text" size="25" />
                                    <input name="wppb_auth_verify_button" id="wppb_auth_verify_button" value="<?php echo esc_html__( 'Check', 'profile-builder' ); ?>" type="button" class="button" />
                                    <input name="wppb_auth_verify_indicator" id="wppb_auth_verify_indicator" value="&nbsp" type="button" class="button" disabled />
                                    <input type="hidden" value="" name="wppb_auth_verify_result" id="wppb_auth_verify_result" />
                                </td>
                            </tr>

                            <?php
                            // Backup Codes Section
                            $remaining_codes = $this->get_remaining_backup_codes_count( $user_id );
                            ?>

                            <tr class="wppb_auth_active wppb_auth_backup">
                                <th><label for="wppb_auth_backup"><?php echo esc_html__( 'Backup Codes', 'profile-builder' ); ?></label></th>
                                <td id="wppb_auth_active">
                                    <input name="wppb_auth_backup" id="wppb_auth_backup" type="hidden" />
                                    <input name="wppb_auth_backup_button" id="wppb_auth_backup_button" value="<?php echo esc_html__( 'Regenerate Backup Codes', 'profile-builder' ); ?>" type="button" class="button" />
                            <?php
                                    if ( $remaining_codes <= 3 && $remaining_codes > 0 ) {
                                        echo '<input name="wppb_auth_backup_indicator" id="wppb_auth_backup_indicator" value="' . esc_html( sprintf( __( '%d remaining codes!', 'profile-builder' ), $remaining_codes ) ) . '" type="button" class="button low" disabled />';
                                    } elseif ( $remaining_codes == 0 ) {
                                        echo '<input name="wppb_auth_backup_indicator" id="wppb_auth_backup_indicator" value="' . esc_html( sprintf( __( 'No remaining codes!', 'profile-builder' ), $remaining_codes ) ) . '" type="button" class="button zero" disabled />';
                                    } else {
                                        echo '<input name="wppb_auth_backup_indicator" id="wppb_auth_backup_indicator" value="' . esc_html( sprintf( __( '%d remaining codes', 'profile-builder' ), $remaining_codes ) ) . '" type="button" class="button" disabled />';
                                    }
                            ?>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                <?php
                return;
            }
        }
    }

    function handle_backend_edit_profile_update($field, $user_id, $request_data, $form_location ) {

        if( $field['field'] === 'Two-Factor Authentication' ) {

            // Check if 2FA is forced for this user
            $is_forced = $this->is_2fa_forced_for_user( $user_id );
            $grace_period_expired = $this->is_grace_period_expired( $user_id );
            $current_2fa_enabled = get_user_option( 'wppb_auth_enabled', $user_id );
            $new_2fa_enabled = !empty($request_data['wppb_auth_enabled']) ? 'enabled' : 'disabled';

            // If 2FA is forced and grace period has expired, user must enable 2FA
            if ( $is_forced && $grace_period_expired && $new_2fa_enabled !== 'enabled' ) {
                add_action( 'admin_notices', function() {
                    echo '<div class="notice notice-error"><p><strong>' . esc_html__( 'Two-Factor Authentication is required for your role. Please enable it.', 'profile-builder' ) . '</strong></p></div>';
                });
                return;
            }

            // If 2FA is currently enabled and user is forced, don't allow disabling
            if ( $is_forced && $current_2fa_enabled === 'enabled' && $new_2fa_enabled !== 'enabled' ) {
                add_action( 'admin_notices', function() {
                    echo '<div class="notice notice-error"><p><strong>' . esc_html__( 'You cannot disable Two-Factor Authentication as it is required for your role.', 'profile-builder' ) . '</strong></p></div>';
                });
                return;
            }

            update_user_option($user_id, 'wppb_auth_enabled', $new_2fa_enabled, true);
            update_user_option($user_id, 'wppb_auth_relaxedmode',
                empty($request_data['wppb_auth_relaxedmode']) ? 'disabled' : 'enabled', true);
            update_user_option($user_id, 'wppb_auth_description',
                trim(sanitize_text_field($request_data['wppb_auth_description'])), true);
            update_user_option($user_id, 'wppb_auth_secret',
                trim($request_data['wppb_auth_secret']), true);

            // Generate backup codes when 2FA is enabled for the first time
            if ( $current_2fa_enabled !== 'enabled' && $new_2fa_enabled === 'enabled' ) {
                $existing_codes = get_user_option( 'wppb_auth_backup_codes', $user_id );
                if ( !$existing_codes || empty( $existing_codes ) ) {
                    $codes = $this->generate_backup_codes( 10 );
                    $this->save_backup_codes( $user_id, $codes['hashed'] );
                }
            }

            // If user just enabled 2FA and was redirected here, redirect them back to intended destination
            if ( $current_2fa_enabled !== 'enabled' && $new_2fa_enabled === 'enabled' && isset( $_GET['wppb_2fa_required'] ) ) {
                $redirect_url = get_transient( 'wppb_2fa_redirect_' . $user_id );
                if ( $redirect_url ) {
                    delete_transient( 'wppb_2fa_redirect_' . $user_id );
                    add_action( 'admin_notices', function() use ($redirect_url) {
                        echo '<div class="notice notice-success is-dismissible">';
                        echo '<p><strong>' . esc_html__( 'Two-Factor Authentication has been successfully set up!', 'profile-builder' ) . '</strong></p>';
                        echo '<p><a href="' . esc_url( $redirect_url ) . '" class="button button-primary">' . esc_html__( 'Continue to your destination', 'profile-builder' ) . '</a></p>';
                        echo '</div>';
                    });
                }
            }
        }
    }

    /**
     * Add 2FA settings field to the PB field validation sequence
     */
    function add_field_to_pb_field_validation_sequence($manage_fields, $args ){
        if( ( isset( $args['form_type'] ) && $args['form_type'] === 'edit_profile' ) || ( isset( $args['context'] ) && $args['context'] === 'validate_backend' ) ) {
            $wppb_two_factor_authentication_settings = get_option( 'wppb_two_factor_authentication_settings', 'not_found' );
            if( isset( $wppb_two_factor_authentication_settings['enabled'] ) && $wppb_two_factor_authentication_settings['enabled'] === 'yes' ) {

                // Check if we should hide other fields when redirected for 2FA setup
                // If fields are hidden on frontend, we must also exclude them from validation
                $hide_other_fields = false;
                if ( isset( $_GET['wppb_2fa_required'] ) && $_GET['wppb_2fa_required'] === '1' ) {
                    if ( apply_filters( 'wppb_2fa_hide_other_fields_on_redirect', true ) ) {
                        $hide_other_fields = true;
                    }
                }

                if ( $hide_other_fields ) {
                    // Only validate the 2FA field
                    return $this->add_field_info( array() );
                } else {
                    return $this->add_field_info($manage_fields);
                }
            }
        }

        return $manage_fields;
    }

    /**
     * 2FA settings field validation
     */
    function form_field_validation($message, $field, $request_data, $form_location ){

        if($field['field'] === 'Two-Factor Authentication') {
            // Try to get user ID from various sources
            $user_id = null;

            // Try to get user ID from edit_user parameter (for admins editing other users)
            if ( isset( $request_data['edit_user'] ) && !empty( $request_data['edit_user'] ) ) {
                $user_id = absint( $request_data['edit_user'] );
            }
            // Try to get user by email if available
            elseif ( isset( $request_data['email'] ) && !empty( $request_data['email'] ) ) {
                $user = get_user_by( 'email', $request_data['email'] );
                if ( $user ) {
                    $user_id = $user->ID;
                }
            }
            // Fall back to current user
            else {
                $current_user = wp_get_current_user();
                if ( $current_user && $current_user->ID ) {
                    $user_id = $current_user->ID;
                }
            }

            if ( !$user_id ) {
                return $message;
            }

            // Get current 2FA status
            $current_2fa_enabled = get_user_option( 'wppb_auth_enabled', $user_id );

            // Get 2FA settings
            $wppb_two_factor_authentication_settings = get_option( 'wppb_two_factor_authentication_settings', 'not_found' );

            // Check if TOTP verification is required on profile edit (for users with 2FA enabled)
            if ( $current_2fa_enabled === 'enabled' ) {
                $require_totp_profile_edit = 'yes';
                if ( $wppb_two_factor_authentication_settings !== 'not_found' && is_array( $wppb_two_factor_authentication_settings ) && isset( $wppb_two_factor_authentication_settings['require_totp_profile_edit'] ) ) {
                    $require_totp_profile_edit = $wppb_two_factor_authentication_settings['require_totp_profile_edit'];
                }

                // If setting requires TOTP on profile edit, verify TOTP is valid
                if ( $require_totp_profile_edit === 'yes' ) {
                    if ( !isset( $request_data['wppb_auth_verify_result'] ) || $request_data['wppb_auth_verify_result'] !== 'valid' ) {
                        return __( 'Please verify TOTP to edit your profile.', 'profile-builder' );
                    }
                }
            }

            // Validate 2FA settings changes (if user is modifying 2FA settings)
            $is_changing_2fa_settings = isset( $request_data['wppb_auth_enabled'] );

            if ( $is_changing_2fa_settings ) {
                $new_2fa_enabled = !empty( $request_data['wppb_auth_enabled'] ) ? 'on' : '';

                // Check if 2FA is forced for this user (only needed when changing settings)
                $is_forced = $this->is_2fa_forced_for_user( $user_id );

                // If 2FA is currently enabled and user is forced, don't allow disabling
                if ( $is_forced && $current_2fa_enabled === 'enabled' && $new_2fa_enabled !== 'on' ) {
                    return __( 'You cannot disable Two-Factor Authentication as it is required for your role.', 'profile-builder' );
                }

                // If user is enabling 2FA and grace period has expired, require it
                if ( $is_forced ) {
                    $grace_period_expired = $this->is_grace_period_expired( $user_id );
                    if ( $grace_period_expired && $current_2fa_enabled !== 'enabled' && $new_2fa_enabled !== 'on' ) {
                        return __( 'Two-Factor Authentication is required for your role. Please enable it.', 'profile-builder' );
                    }
                }

                // Validate TOTP verification when enabling or changing 2FA settings
                if ( $new_2fa_enabled === 'on' ) {
                    // Check if settings are being changed (requires TOTP verification)
                    $is_changing_settings = (
                        ( empty( $request_data['wppb_auth_enabled'] ) ? 'disabled' : 'enabled' ) !== get_user_option( 'wppb_auth_enabled', $user_id ) ||
                        ( empty( $request_data['wppb_auth_relaxedmode'] ) ? 'disabled' : 'enabled' ) !== get_user_option( 'wppb_auth_relaxedmode', $user_id ) ||
                        ( trim( sanitize_text_field( $request_data['wppb_auth_description'] ) ) ) !== get_user_option( 'wppb_auth_description', $user_id ) ||
                        ( trim( $request_data['wppb_auth_secret'] ) ) !== get_user_option( 'wppb_auth_secret', $user_id )
                    );

                    // Require TOTP verification if enabling for first time or changing settings
                    if ( $current_2fa_enabled !== 'enabled' || $is_changing_settings ) {
                        if ( !isset( $request_data['wppb_auth_verify_result'] ) || $request_data['wppb_auth_verify_result'] !== 'valid' ) {
                            if ( $current_2fa_enabled !== 'enabled' ) {
                                return __( 'Please verify TOTP to set up Two-Factor Authentication', 'profile-builder' );
                            } else {
                                return __( 'Please verify TOTP to change Two-Factor Authentication settings', 'profile-builder' );
                            }
                        }
                    }
                }
            }
        }
        return $message;
    }

    /**
     * Create a new random secret
     */
    function create_secret( ) {
        $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567'; // allowed characters in Base32
        $secret = '';
        for ( $i = 0; $i < 16; $i++ ) {
            $secret .= $chars[wp_rand(0, strlen($chars) - 1)];
        }
        return $secret;
    }

    /**
     * AJAX callback function used to add field to the login form when necessary
     */
    function ajax_add_auth_field_to_login_form() {
        // Some AJAX security.
        check_ajax_referer( 'WPPBAuth_field_on_login_form', 'nonce' );

        if ( isset( $_REQUEST['user'] )) {

            $username = sanitize_text_field( $_REQUEST['user'] );

            if ( is_email( $username ) ) {
                $userdata = get_user_by('email', $username);
            } else {
                $userdata = get_user_by('login', $username);
            }

            $wppb_two_factor_authentication_settings = get_option('wppb_two_factor_authentication_settings', 'not_found');

            header('Content-Type: application/json');

            if ($userdata instanceof WP_User &&
                isset($wppb_two_factor_authentication_settings['enabled'], $wppb_two_factor_authentication_settings['roles']) &&
                $wppb_two_factor_authentication_settings['enabled'] === 'yes' &&
                get_user_option('wppb_auth_enabled', $userdata->ID) === 'enabled') {

                if ($this->should_user_see_field($wppb_two_factor_authentication_settings['roles'], $userdata)) {
                    $result = array(
                        'field'  => $this->auth_code_field(),
                        'notice' => isset( $_REQUEST['location'] ) && $_REQUEST['location'] === 'backend' ? $this->input_TOTP_alert_back() : $this->input_TOTP_alert_front()
                    );
                    echo json_encode($result);
                    die();
                }
            }
        }


        echo json_encode( false );
        die();
    }

    /**
     * AJAX callback function used to generate new secret
     */
    function ajax_new_secret() {
        // Some AJAX security.
        check_ajax_referer( 'WPPBAuthaction', 'nonce' );

        // Create new secret.
        $secret = $this->create_secret( );

        $result = array( 'new-secret' => $secret );
        header( 'Content-Type: application/json' );
        echo json_encode( $result );

        die( );
    }

    /**
     * AJAX callback function used to validate TOTP
     */
    function ajax_check_code() {
        global $user_id;
        $valid = false;

        // AJAX security.
        check_ajax_referer( 'WPPBAuthaction', 'nonce' );

        if ( isset( $_REQUEST['secret'] ) ) {

            // Get the user's secret
            $secret = sanitize_text_field($_REQUEST['secret']);

            // Figure out if user is using relaxed mode
            $relaxedmode = '';
            if ( isset( $_REQUEST['relaxedmode'] ) ){
                $relaxedmode = sanitize_text_field($_REQUEST['relaxedmode']);
            }

            // Get the verification code entered by the user trying to login
            if (!empty($_REQUEST['otp'])) { // Prevent PHP notices when using app password login
                $otp = trim(sanitize_text_field($_REQUEST['otp']));
            } else {
                $otp = '';
            }
            // When was the last successful login performed ?
            $lasttimeslot = trim(get_user_option('wppb_auth_lasttimeslot', $user_id));
            // Valid code ?
            if ($timeslot = $this->verify($secret, $otp, $relaxedmode, $lasttimeslot)) {
                // Store the timeslot in which login was successful.
                update_user_option($user_id, 'wppb_auth_lasttimeslot', $timeslot, true);
                $valid = true;
            }
        }

        $result = array( 'valid-otp' => $valid );
        header( 'Content-Type: application/json' );
        echo json_encode( $result );

        die();
    }


    /**
     * Handle error for the frontend Login form
     */
    function login_error_message_handler( $userdata, $password )
    {
        $wppb_two_factor_authentication_settings = get_option( 'wppb_two_factor_authentication_settings', 'not_found' );

        if ( !is_wp_error( $userdata ) &&
            isset( $wppb_two_factor_authentication_settings['enabled'], $wppb_two_factor_authentication_settings['roles'] ) && $wppb_two_factor_authentication_settings['enabled'] === 'yes' &&
            get_user_option( 'wppb_auth_enabled', $userdata->ID ) === 'enabled' ) {

            if ( $this->should_user_see_field( $wppb_two_factor_authentication_settings['roles'], $userdata ) ) {
                if ( !isset( $_POST['auth'] ) || empty( $_POST['auth'] ) ) {
                    $errorMessage = __( 'Please enter the code from your Authenticator app.', 'profile-builder' );
                    return new WP_Error( 'wppb_login_auth', $errorMessage );
                }
                if ( !$this->check_otp( $userdata, $userdata->data->user_login, $password ) ) {
                    $errorMessage = '<strong>' . __( 'ERROR:', 'profile-builder' ) . '</strong> ' . __( 'Your Authenticator code was incorrect. Please try again.', 'profile-builder' );
                    return new WP_Error( 'wppb_login_auth', $errorMessage );
                }
            }
        }
        return $userdata;
    }

    function should_user_see_field( $two_factor_authentication_roles, $userdata ) {
        if (in_array('*', $two_factor_authentication_roles, true) ){
            return true;
        }

        foreach ($two_factor_authentication_roles as $key => $value) {
            if ( in_array($value, $userdata->roles, true) ) {
                return true;
            }
        }
        return false;
    }

    /**
     * Add script that dynamically adds field to frontend Login form
     */
    function add_field_to_frontend_login_form($form_part, $args ) {
        if( !wp_script_is('jquery', 'done') && !is_admin() ){
            wp_print_scripts('jquery');
        }

        return '
            <script type="text/javascript">
                jQuery( document ).ready(function() {
                    var WPPBAuthNonce = "' . wp_create_nonce( 'WPPBAuth_field_on_login_form' ) . '";
                    var ajaxurl = "' . admin_url( 'admin-ajax.php' ) . '";
    
                    if ( !jQuery(".login-auth").length ){
                    
                        jQuery("#wppb-loginform").one("submit", wppb2fa_login_form_submit_handler);
                        jQuery(document).one("wppb_invisible_recaptcha_success wppb_v3_recaptcha_success", wppb2fa_login_form_submit_handler);

                        function wppb2fa_login_form_submit_handler(event) {
                            
                            var thisForm = this;

                            if( event.type && ( event.type === "wppb_invisible_recaptcha_success" || event.type === "wppb_v3_recaptcha_success" ) ) {
                                thisForm = jQuery("#wppb-loginform");
                            }

                            event.preventDefault();

                            var data = new Object();
                                data["action"]	 = "WPPBAuth_field_on_login_form";
                                data["nonce"]	 = WPPBAuthNonce;
                                data["location"] = "frontend";
                                data["user"]	 = jQuery("#wppb_user_login.input").val();
                            jQuery.post(ajaxurl, data, function(response) {

                                // Add class to form
                                jQuery(thisForm).addClass("wppb-2fa-authentication-requested");

                                if ( response && !jQuery(".login-auth").length ) {
                                    jQuery("#wppb-login-wrap").before(response["notice"]);
                                    jQuery(".login-password").after(response["field"]);
                                    
                                    // include dynamically added field into FD Styles 
                                    if ( typeof handleFloatingLabels === "function" ) {
                                        handleFloatingLabels( jQuery( ".login-auth" ) );
                                    }
                                    
                                    // maybe add placeholder to the 2FA Field
                                    if ( typeof loginAuthPlaceholder === "function" ) {
                                        loginAuthPlaceholder( jQuery( ".login-auth" ) );
                                    }                                    
                                                                        
                                } else {
                                    if( !event.type || ( event.type !== "wppb_invisible_recaptcha_success" && event.type !== "wppb_v3_recaptcha_success" ) ) {
                                        jQuery("#wppb-loginform").submit();
                                    }
                                }
                            });
                        }
                    }
                });
            </script>
            ';
    }

	/**
	 * Add script that dynamically adds field to backend Login form
	 */
	function add_field_to_backend_login_form( ) {
		if( !wp_script_is('jquery', 'done') && !is_admin() ){
			wp_print_scripts('jquery');
		}

		echo '
            <script type="text/javascript">
                jQuery( document ).ready(function() {
                    var WPPBAuthNonce = "' . esc_html( wp_create_nonce( 'WPPBAuth_field_on_login_form' ) ) . '";
                    var ajaxurl = "' . esc_html( admin_url( 'admin-ajax.php' ) ) . '";
    
                    if ( !jQuery(".login-auth").length ){
                        jQuery("#loginform").one("submit", function(event) {
                            var thisForm = this;
                            event.preventDefault();
    
                            var data = new Object();
                            data["action"]	= "WPPBAuth_field_on_login_form";
                            data["nonce"]	= WPPBAuthNonce;
                            data["location"]= "backend";
                            data["user"]	= jQuery("#wppb_user_login.input").val();
                            jQuery.post(ajaxurl, data, function(response) {
                                if ( response ){
                                    jQuery("#loginform").before(response["notice"]);
                                    jQuery("#loginform .user-pass-wrap").after(response["field"]);
                                } else {
                                    thisForm.submit();
                                }
                            });
                        });
                    }
                });
            </script>
            ';
	}

    /**
     * Handle error for the backend Login form
     */
    function back_end_errors_filter( $errors, $redirect_to ) {
        if ( isset( $errors->errors['wppb_login_auth'] ) ) {
            add_action( 'login_form', array( $this, 'echo_auth_code_field' ) );
        }
        return $errors;
    }

    function input_TOTP_alert_front() {
        return apply_filters( 'wppb_login_2fa_otp_validation_message','<p class="wppb-warning">' . __( 'Please enter the code from your Authenticator app.', 'profile-builder' ) . '</p>' );
    }

	function input_TOTP_alert_back() {
		return '<div id="login_error">' . __( 'Please enter the code from your Authenticator app.', 'profile-builder' ) . '<br></div>';
	}

    /**
     * Echo field HTML for the backend Login form
     */
    function echo_auth_code_field() {
        echo $this->auth_code_field(); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    }

    /**
     * Field HTML for Login forms
     */
    function auth_code_field( $form_part = '' ) {

        $forms_settings = get_option( 'wppb_toolbox_forms_settings' );

        if ( is_array( $forms_settings ) && isset( $forms_settings['placeholder-labels'] ) && $forms_settings['placeholder-labels'] === 'yes' ) {
            $label_position_class = 'label-inside';
        } else {
            $label_position_class = 'label-outside';
        }

        $form_part .= '
			<p class="wppb-form-field wppb-form-text-field login-auth '. $label_position_class .'">' . $this->auth_code_field_inner() . '
			</p>';

        return $form_part;

    }

    /**
     * Field inner HTML for Login forms
     */
    function auth_code_field_inner( ) {
        return '
				<label for="login_auth">' . __( 'Authentication Code', 'profile-builder' ) . '</label>
				<input type="text" name="auth" id="login_auth" class="input" value="" size="20" autocomplete="off" />
				<p class="description">' . __( 'Enter your 6-digit authenticator code or 16-digit backup code.', 'profile-builder' ) . '</p>';
    }

    /**
     * TOTP check on login attempt
     */
    function check_otp ( $user, $username = '', $password = '' ) {
        // Get the verification code
        if ( !empty( $_POST['auth'] ) ) {
            $otp = trim( sanitize_text_field($_POST['auth']) );
        } else {
            $otp = '';
        }

        // Check format: 16 digits = backup code, 6 digits = TOTP
        if ( strlen( $otp ) === 16 && ctype_digit( $otp ) ) {
            // Validate as backup code
            if ( $this->verify_backup_code( $user->ID, $otp ) ) {
                return true;
            }
        } elseif ( strlen( $otp ) === 6 ) {
            // Existing TOTP validation
            $secret = trim( get_user_option( 'wppb_auth_secret', $user->ID ) );
            $relaxedmode = trim( get_user_option( 'wppb_auth_relaxedmode', $user->ID ) );
            $lasttimeslot = trim( get_user_option( 'wppb_auth_lasttimeslot', $user->ID ) );

            if ( $timeslot = $this->verify( $secret, $otp, $relaxedmode, $lasttimeslot ) ) {
                // Save the timeslot in which login was successful
                update_user_option( $user->ID, 'wppb_auth_lasttimeslot', $timeslot, true );
                return true;
            }
        }

        return false;
    }

    /**
     * Verification code check
     */
    function verify( $secretkey, $thistry, $relaxedmode, $lasttimeslot ) {
        if ( strlen( $thistry ) !== 6 ) {
            return false;
        } else {
            $thistry = (int)$thistry;
        }
        // account for time drift
        if ( $relaxedmode === 'enabled' ) {
            $firstcount = -8;
            $lastcount  =  8;
        } else {
            $firstcount = -1;
            $lastcount  =  1;
        }

        $tm = floor( time( ) / 30 );

        $secretkey = WPPB_Base32::decode( $secretkey );
        for ( $i = $firstcount; $i <= $lastcount; $i++ ) {
            $time=chr( 0 ).chr( 0 ).chr( 0 ).chr( 0 ).pack( 'N*', $tm+$i );
            $hm = hash_hmac( 'SHA1', $time, $secretkey, true );
            $offset = ord( substr( $hm, -1 ) ) & 0x0F;
            $hashpart=substr( $hm, $offset, 4 );
            $value=unpack( "N", $hashpart );
            $value=$value[1];
            $value &= 0x7FFFFFFF;
            $value %= 1000000;
            if ( $value === $thistry ) {
                // Current login attempt must not happen before the last successful login
                if ( $lasttimeslot >= ( $tm+$i ) ) {
                    return false;
                }
                // Return timeslot in which login happened.
                return $tm+$i;
            }
        }
        return false;
    }

    /**
     * Get pages that contain Edit Profile shortcode or block
     * Results are cached for 1 hour using transient
     *
     * @return array Array of page objects with ID and title
     */
    static function get_edit_profile_pages() {
        // Check transient
        $cached_pages = get_transient( 'wppb_2fa_edit_profile_pages' );
        if ( $cached_pages !== false ) {
            return $cached_pages;
        }

        global $wpdb;

        $edit_profile_pages = array();

        // Query for pages containing the shortcode or block
        // We use a direct query for performance to avoid loading all pages
        $query = "
            SELECT ID, post_title 
            FROM {$wpdb->posts} 
            WHERE post_type = 'page' 
            AND post_status = 'publish' 
            AND (post_content LIKE %s OR post_content LIKE %s)
        ";

        $like_shortcode = '%' . $wpdb->esc_like( '[wppb-edit-profile' ) . '%';
        $like_block = '%' . $wpdb->esc_like( '<!-- wp:wppb/edit-profile' ) . '%';

        $results = $wpdb->get_results( $wpdb->prepare( $query, $like_shortcode, $like_block ) );

        if ( $results ) {
            foreach ( $results as $page ) {
                $edit_profile_pages[] = array(
                    'ID' => $page->ID,
                    'title' => $page->post_title
                );
            }
        }

        // Cache results
        set_transient( 'wppb_2fa_edit_profile_pages', $edit_profile_pages, HOUR_IN_SECONDS );

        return $edit_profile_pages;
    }

    /**
     * Clear the edit profile pages cache
     * Called when pages are saved/updated
     */
    function clear_edit_profile_pages_cache( $post_id ) {
        // Only clear cache for pages
        if ( get_post_type( $post_id ) === 'page' ) {
            delete_transient( 'wppb_2fa_edit_profile_pages' );
        }
    }

    function wppb_two_factor_authentication_settings_update() {

        if( !isset( $_POST['option_page'] ) || $_POST['option_page'] != 'wppb_general_settings' )
            return;
    
        if( ( !is_multisite() && current_user_can( 'manage_options' ) ) || ( is_multisite() && current_user_can( 'manage_network' ) ) ){
    
            if ( isset( $_POST['wppb_two_factor_authentication_settings'] ) ) {
                $settings = $_POST['wppb_two_factor_authentication_settings']; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

                // Sanitize enabled setting
                if ( isset( $settings['enabled'] ) ) {
                    $settings['enabled'] = sanitize_text_field( $settings['enabled'] );
                }

                // Sanitize roles array
                if ( isset( $settings['roles'] ) && is_array( $settings['roles'] ) ) {
                    $settings['roles'] = array_map( 'sanitize_text_field', $settings['roles'] );
                }

                // Sanitize force 2FA settings
                if ( isset( $settings['force_2fa_enabled'] ) ) {
                    $settings['force_2fa_enabled'] = sanitize_text_field( $settings['force_2fa_enabled'] );

                    // Validate that Edit Profile pages exist if Force 2FA is being enabled
                    if ( $settings['force_2fa_enabled'] === 'yes' ) {
                        $edit_profile_pages = self::get_edit_profile_pages();
                        if ( empty( $edit_profile_pages ) ) {
                            // Prevent enabling Force 2FA if no Edit Profile pages exist
                            $settings['force_2fa_enabled'] = 'no';
                        }
                    }
                } else {
                    $settings['force_2fa_enabled'] = 'no';
                }

                if ( isset( $settings['force_2fa_roles'] ) && is_array( $settings['force_2fa_roles'] ) ) {
                    $settings['force_2fa_roles'] = array_map( 'sanitize_text_field', $settings['force_2fa_roles'] );

                    // Clean up force 2FA roles - only keep roles that are enabled for 2FA
                    $enabled_roles = ( isset( $settings['roles'] ) && is_array( $settings['roles'] ) ) ? $settings['roles'] : array();
                    if ( !in_array( '*', $enabled_roles, true ) ) {
                        // If not all roles are enabled, filter force 2FA roles to only include enabled ones
                        $settings['force_2fa_roles'] = array_intersect( $settings['force_2fa_roles'], $enabled_roles );
                    }
                } else {
                    $settings['force_2fa_roles'] = array();
                }

                if ( isset( $settings['grace_period_days'] ) ) {
                    $settings['grace_period_days'] = max( 0, min( 365, intval( $settings['grace_period_days'] ) ) );
                } else {
                    $settings['grace_period_days'] = 0;
                }

                if ( isset( $settings['require_totp_profile_edit'] ) ) {
                    $settings['require_totp_profile_edit'] = sanitize_text_field( $settings['require_totp_profile_edit'] );
                } else {
                    // If checkbox is unchecked (not in POST), check if setting exists in DB
                    $existing_settings = get_option( 'wppb_two_factor_authentication_settings', array() );
                    if ( isset( $existing_settings['require_totp_profile_edit'] ) ) {
                        // Setting exists, user explicitly unchecked it, so save as 'no'
                        $settings['require_totp_profile_edit'] = 'no';
                    } else {
                        // Setting doesn't exist yet, default to 'yes'
                        $settings['require_totp_profile_edit'] = 'yes';
                    }
                }

                // Sanitize edit profile page ID setting
                if ( isset( $settings['edit_profile_page_id'] ) ) {
                    $settings['edit_profile_page_id'] = sanitize_text_field( $settings['edit_profile_page_id'] );
                    // Validate that the page ID exists if it's not 'admin'
                    if ( $settings['edit_profile_page_id'] !== 'admin' && $settings['edit_profile_page_id'] !== '0' ) {
                        $page_id = intval( $settings['edit_profile_page_id'] );
                        if ( $page_id > 0 && get_post_type( $page_id ) !== 'page' ) {
                            // Invalid page ID, reset to admin
                            $settings['edit_profile_page_id'] = 'admin';
                        } else {
                            $settings['edit_profile_page_id'] = $page_id;
                        }
                    }
                } else {
                    $settings['edit_profile_page_id'] = 'admin';
                }
                update_option( 'wppb_two_factor_authentication_settings', $settings );
            } else {
                update_option( 'wppb_two_factor_authentication_settings', array(
                    'enabled' => 'no',
                    'roles' => array(),
                    'force_2fa_enabled' => 'no',
                    'force_2fa_roles' => array(),
                    'grace_period_days' => 0,
                    'require_totp_profile_edit' => 'yes',
                    'edit_profile_page_id' => 'admin',

                ) );
            }

        }

    }

    /**
     * Check if 2FA is forced for a specific user based on their role
     *
     * @param int $user_id The user ID to check
     * @return bool True if 2FA is forced for this user
     */
    function is_2fa_forced_for_user( $user_id ) {
        $wppb_two_factor_authentication_settings = get_option( 'wppb_two_factor_authentication_settings', 'not_found' );

        if ( !isset( $wppb_two_factor_authentication_settings['force_2fa_enabled'] ) ||
             $wppb_two_factor_authentication_settings['force_2fa_enabled'] !== 'yes' ) {
            return false;
        }

        $user = get_userdata( $user_id );
        if ( !$user ) {
            return false;
        }

        // Check if user has any role that requires forced 2FA
        $force_2fa_roles = isset( $wppb_two_factor_authentication_settings['force_2fa_roles'] ) ?
                           $wppb_two_factor_authentication_settings['force_2fa_roles'] : array();

        foreach ( $user->roles as $role ) {
            if ( in_array( $role, $force_2fa_roles, true ) ) {
                return true;
            }
        }

        return false;
    }

    /**
     * Set grace period for a user
     *
     * @param int $user_id The user ID
     * @return bool True on success, false on failure
     */
    function set_grace_period( $user_id ) {
        // Store the start time so the end time can be calculated dynamically based on global settings
        return update_user_option( $user_id, 'wppb_2fa_grace_period_start', time(), true );
    }

    /**
     * Get grace period end timestamp for a user
     *
     * @param int $user_id The user ID
     * @return int|false Timestamp or false if no grace period
     */
    function get_grace_period_end( $user_id ) {
        $wppb_two_factor_authentication_settings = get_option( 'wppb_two_factor_authentication_settings', 'not_found' );
        $grace_period_days = isset( $wppb_two_factor_authentication_settings['grace_period_days'] ) ?
                            intval( $wppb_two_factor_authentication_settings['grace_period_days'] ) : 0;

        if ( $grace_period_days <= 0 ) {
            return false;
        }

        $grace_period_start = get_user_option( 'wppb_2fa_grace_period_start', $user_id );

        // If no start time is set, but the user is forced to use 2FA,
        // initialize the grace period starting now. This handles existing users
        // who become forced via settings changes.
        if ( ! $grace_period_start && $this->is_2fa_forced_for_user( $user_id ) ) {
            $this->set_grace_period( $user_id );
            $grace_period_start = time();
        }

        if ( ! $grace_period_start ) {
            return false;
        }

        return intval( $grace_period_start ) + ( $grace_period_days * DAY_IN_SECONDS );
    }

    /**
     * Check if the grace period has expired for a user
     *
     * @param int $user_id The user ID
     * @return bool True if grace period has expired or doesn't exist
     */
    function is_grace_period_expired( $user_id ) {
        $grace_period_end = $this->get_grace_period_end( $user_id );

        if ( !$grace_period_end ) {
            return true; // No grace period means expired
        }

        return time() > $grace_period_end;
    }

    /**
     * Get remaining grace period days for a user
     *
     * @param int $user_id The user ID
     * @return int Number of days remaining in grace period (0 if expired)
     */
    function get_remaining_grace_period_days( $user_id ) {
        $grace_period_end = $this->get_grace_period_end( $user_id );

        if ( !$grace_period_end ) {
            return 0;
        }

        $remaining_seconds = $grace_period_end - time();
        return max( 0, ceil( $remaining_seconds / DAY_IN_SECONDS ) );
    }

    /**
     * Update forced 2FA status for a user when their role changes
     *
     * @param int $user_id The user ID
     * @param string $role The new role
     * @param array $old_roles The old roles
     */
    function update_forced_2fa_status_on_role_change( $user_id, $role, $old_roles ) {
        $is_forced = $this->is_2fa_forced_for_user( $user_id );

        if ( $is_forced ) {
            update_user_option( $user_id, 'wppb_2fa_forced', 'yes', true );
            $this->set_grace_period( $user_id );
        } else {
            update_user_option( $user_id, 'wppb_2fa_forced', 'no', true );
            delete_user_option( $user_id, 'wppb_2fa_grace_period_end', true );
            delete_user_option( $user_id, 'wppb_2fa_grace_period_start', true );
        }
    }

    /**
     * Handle new user registration - set up forced 2FA if needed
     *
     * @param int $user_id The user ID
     */
    function handle_new_user_registration( $user_id ) {
        if ( $this->is_2fa_forced_for_user( $user_id ) ) {
            update_user_option( $user_id, 'wppb_2fa_forced', 'yes', true );
            $this->set_grace_period( $user_id );
        }
    }

    /**
     * Handle user role change - update forced 2FA status
     *
     * @param int $user_id The user ID
     * @param string $role The new role
     * @param array $old_roles The old roles
     */
    function handle_user_role_change( $user_id, $role, $old_roles ) {
        $this->update_forced_2fa_status_on_role_change( $user_id, $role, $old_roles );
    }

    /**
     * Handle login redirect for forced 2FA enforcement
     *
     * @param string $redirect_to The redirect URL
     * @param string $requested_redirect_to The originally requested redirect URL
     * @param WP_User|WP_Error $user The user object
     * @return string The redirect URL
     */
    function handle_forced_2fa_login_redirect( $redirect_to, $requested_redirect_to, $user ) {
        // Only apply to successful logins
        if ( is_wp_error( $user ) || !$user ) {
            return $redirect_to;
        }

        // Check if user is subject to forced 2FA
        if ( !$this->is_2fa_forced_for_user( $user->ID ) ) {
            return $redirect_to;
        }

        // Check if user has 2FA enabled
        $user_2fa_enabled = get_user_option( 'wppb_auth_enabled', $user->ID );
        if ( $user_2fa_enabled === 'enabled' ) {
            return $redirect_to;
        }

        // Check if grace period has expired
        if ( !$this->is_grace_period_expired( $user->ID ) ) {
            return $redirect_to;
        }

        // Store the intended redirect URL for later
        set_transient( 'wppb_2fa_redirect_' . $user->ID, $redirect_to, HOUR_IN_SECONDS );

        // Get the selected edit profile page from settings
        $wppb_two_factor_authentication_settings = get_option( 'wppb_two_factor_authentication_settings', 'not_found' );
        $edit_profile_page_id = 'admin';

        if ( $wppb_two_factor_authentication_settings !== 'not_found' &&
             is_array( $wppb_two_factor_authentication_settings ) &&
             isset( $wppb_two_factor_authentication_settings['edit_profile_page_id'] ) ) {
            $edit_profile_page_id = $wppb_two_factor_authentication_settings['edit_profile_page_id'];
        }

        // Use selected page if it's set and valid
        if ( $edit_profile_page_id !== 'admin' && $edit_profile_page_id !== '0' ) {
            $page_id = intval( $edit_profile_page_id );
            if ( $page_id > 0 && get_post_type( $page_id ) === 'page' ) {
                $profile_url = get_permalink( $page_id );
                return add_query_arg( array( 'wppb_2fa_required' => '1' ), $profile_url );
            }
        }

        // No page selected, invalid page, or 'admin' selected - try to find first available Edit Profile page
        // Only search for frontend pages if we haven't explicitly selected 'admin'
        if ( $edit_profile_page_id !== 'admin' ) {
            $edit_profile_pages = self::get_edit_profile_pages();
            if ( !empty( $edit_profile_pages ) ) {
                $profile_url = get_permalink( $edit_profile_pages[0]['ID'] );
                return add_query_arg( array( 'wppb_2fa_required' => '1' ), $profile_url );
            }
        }

        // Fallback to admin profile page
        $profile_url = admin_url( 'profile.php' );
        if ( is_multisite() ) {
            $profile_url = get_dashboard_url( $user->ID, 'profile.php' );
        }

        return add_query_arg( array( 'wppb_2fa_required' => '1' ), $profile_url );
    }

    /**
     * Show admin notices for forced 2FA requirements
     */
    function show_forced_2fa_admin_notices() {
        $current_user = wp_get_current_user();
        if ( !$current_user || !$current_user->ID ) {
            return;
        }

        // Check if user is subject to forced 2FA
        if ( !$this->is_2fa_forced_for_user( $current_user->ID ) ) {
            return;
        }

        // Check if user has 2FA enabled
        $user_2fa_enabled = get_user_option( 'wppb_auth_enabled', $current_user->ID );
        if ( $user_2fa_enabled === 'enabled' ) {
            return;
        }

        // Check if grace period has expired
        $grace_period_expired = $this->is_grace_period_expired( $current_user->ID );
        $remaining_days = $this->get_remaining_grace_period_days( $current_user->ID );

        $notice_class = 'notice notice-error';
        $is_dismissible = 'is-dismissible';

        if ( !$grace_period_expired && $remaining_days > 0 ) {
            $notice_class = 'notice notice-warning';
            $notice_message = sprintf(
                __( 'Your role requires Two-Factor Authentication. You have %d days remaining to set it up.', 'profile-builder' ),
                $remaining_days
            );
        } else {
            $notice_message = __( 'Your role requires Two-Factor Authentication. Please set it up immediately.', 'profile-builder' );
            $is_dismissible = ''; // Don't allow dismissing critical notices
        }

        echo '<div class="' . esc_attr( $notice_class ) . ' ' . esc_attr( $is_dismissible ) . '">';
        echo '<p><strong>' . esc_html( $notice_message ) . '</strong></p>';
        echo '<p><a href="' . esc_url( admin_url( 'profile.php#wppb_auth_enabled' ) ) . '" class="button button-primary">' . esc_html__( 'Set up 2FA Now', 'profile-builder' ) . '</a></p>';
        echo '</div>';
    }

    /**
     * Bypass Custom Redirects if 2FA is required
     *
     * @param string $redirect_url The custom redirect URL
     * @param string $type The type of redirect
     * @param string $original_redirect_url The original redirect URL
     * @param object $user The user object
     * @return string|null The redirect URL or null to bypass
     */
    function bypass_custom_redirects_for_2fa( $redirect_url, $type, $original_redirect_url, $user ) {
        // If we're already bypassing or empty, don't interfere
        if ( empty( $redirect_url ) ) {
            return $redirect_url;
        }

        // If user is not passed, try to get current user
        if ( ! $user || is_wp_error( $user ) ) {
            $user = wp_get_current_user();
        }

        if ( ! $user || $user->ID == 0 ) {
            return $redirect_url;
        }

        // Security check: 2FA must be forced for this user to allow any bypass
        $is_forced = $this->is_2fa_forced_for_user( $user->ID );

        if ( ! $is_forced ) {
            return $redirect_url;
        }

        // Check if we are specifically requesting 2FA setup via URL
        if ( isset( $_GET['wppb_2fa_required'] ) ) {
            return null;
        }

        // Check if they have 2FA enabled
        $two_factor_enabled = get_user_option( 'wppb_auth_enabled', $user->ID );

        if ( $two_factor_enabled !== 'enabled' ) {
            // Check grace period
            if ( $this->is_grace_period_expired( $user->ID ) ) {
                // 2FA is required and not enabled, and grace period expired - bypass custom redirects
                return null;
            }
        }

        return $redirect_url;
    }

    /**
     * Handle already logged-in users who need to set up 2FA
     */
    function handle_already_logged_in_users() {
        // Only apply to logged-in users
        if ( !is_user_logged_in() ) {
            return;
        }

        $current_user = wp_get_current_user();
        if ( !$current_user || !$current_user->ID ) {
            return;
        }

        // Check if user is subject to forced 2FA
        if ( !$this->is_2fa_forced_for_user( $current_user->ID ) ) {
            return;
        }

        // Check if user has 2FA enabled
        $user_2fa_enabled = get_user_option( 'wppb_auth_enabled', $current_user->ID );
        if ( $user_2fa_enabled === 'enabled' ) {
            return;
        }

        // Check if grace period has expired
        if ( !$this->is_grace_period_expired( $current_user->ID ) ) {
            return;
        }

        // Skip redirection on certain pages to avoid infinite loops
        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( $_SERVER['REQUEST_URI'] ) : '';

        // Don't redirect if it's an AJAX request
        if ( ( defined( 'DOING_AJAX' ) && DOING_AJAX ) || strpos( $request_uri, 'admin-ajax.php' ) !== false ) {
            return;
        }

        // Don't redirect if already on the 2FA required page (indicated by the query parameter)
        if ( isset( $_GET['wppb_2fa_required'] ) && $_GET['wppb_2fa_required'] === '1' ) {
            return;
        }

        // Don't redirect if we're currently saving a profile update (allow the save to complete first)
        if ( isset( $_SERVER['REQUEST_METHOD'] ) &&
            $_SERVER['REQUEST_METHOD'] === 'POST' &&
            isset( $_POST['action'] ) &&
            $_POST['action'] === 'update' &&
            strpos( $request_uri, 'profile.php' ) !== false ) {
            return;
        }

        // Get settings to check if we need to enforce the parameter
        $enforce_param = apply_filters( 'wppb_2fa_hide_other_fields_on_redirect', true );

        // If we are enforcing the parameter, we skip the "already on page" checks below
        // because we want to force a redirect to append the parameter if it's missing
        if ( ! $enforce_param ) {
            if ( is_admin() ) {
                // Don't redirect if already on profile page
                if ( strpos( $request_uri, 'profile.php' ) !== false ) {
                    return;
                }
            } else {
                // Check if current page is an Edit Profile page
                $current_page_id = get_queried_object_id();
                if ( $current_page_id > 0 ) {
                    // Get the selected edit profile page from settings
                    $edit_profile_page_id = 'admin';

                    if ( $wppb_two_factor_authentication_settings !== 'not_found' &&
                         is_array( $wppb_two_factor_authentication_settings ) &&
                         isset( $wppb_two_factor_authentication_settings['edit_profile_page_id'] ) ) {
                        $edit_profile_page_id = $wppb_two_factor_authentication_settings['edit_profile_page_id'];
                    }

                    // Check if current page matches the selected edit profile page
                    if ( $edit_profile_page_id !== 'admin' && $edit_profile_page_id !== '0' ) {
                        $page_id = intval( $edit_profile_page_id );
                        if ( $current_page_id === $page_id ) {
                            return;
                        }
                    }

                    // Check if current page is one of the Edit Profile pages
                    $edit_profile_pages = self::get_edit_profile_pages();
                    foreach ( $edit_profile_pages as $page ) {
                        if ( $current_page_id === $page['ID'] ) {
                            return;
                        }
                    }
                }

                // Fallback: Don't redirect if already on a profile edit page (by URL string)
                if ( strpos( $request_uri, 'edit-profile' ) !== false ) {
                    return;
                }
            }
        }

        // Redirect to appropriate profile page
        // If user is already in the backend (admin area), keep them there regardless of settings
        if ( is_admin() ) {
            $redirect_url = admin_url( 'profile.php?wppb_2fa_required=1' );
        } else {
            // Get the selected edit profile page from settings
            $wppb_two_factor_authentication_settings = get_option( 'wppb_two_factor_authentication_settings', 'not_found' );
            $edit_profile_page_id = 'admin';

            if ( $wppb_two_factor_authentication_settings !== 'not_found' &&
                 is_array( $wppb_two_factor_authentication_settings ) &&
                 isset( $wppb_two_factor_authentication_settings['edit_profile_page_id'] ) ) {
                $edit_profile_page_id = $wppb_two_factor_authentication_settings['edit_profile_page_id'];
            }

            // Use selected page if it's set and valid
            if ( $edit_profile_page_id !== 'admin' && $edit_profile_page_id !== '0' ) {
                $page_id = intval( $edit_profile_page_id );
                if ( $page_id > 0 && get_post_type( $page_id ) === 'page' ) {
                    $redirect_url = get_permalink( $page_id );
                    $redirect_url = add_query_arg( array( 'wppb_2fa_required' => '1' ), $redirect_url );
                } else {
                    // Invalid page ID, fallback to admin profile page
                    $redirect_url = admin_url( 'profile.php?wppb_2fa_required=1' );
                }
            } else {
                // No page selected or 'admin' selected, try to find first available Edit Profile page
                // Only search for frontend pages if we haven't explicitly selected 'admin'
                if ( $edit_profile_page_id !== 'admin' ) {
                    $edit_profile_pages = self::get_edit_profile_pages();
                    if ( !empty( $edit_profile_pages ) ) {
                        $redirect_url = get_permalink( $edit_profile_pages[0]['ID'] );
                        $redirect_url = add_query_arg( array( 'wppb_2fa_required' => '1' ), $redirect_url );
                    } else {
                        // Fallback to admin profile page
                        $redirect_url = admin_url( 'profile.php?wppb_2fa_required=1' );
                    }
                } else {
                    // Fallback to admin profile page
                    $redirect_url = admin_url( 'profile.php?wppb_2fa_required=1' );
                }
            }
        }

        // Store current URL for later redirect
        $http_host = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( $_SERVER['HTTP_HOST'] ) : '';
        if ( $http_host && $request_uri ) {
            $current_url = ( is_ssl() ? 'https://' : 'http://' ) . $http_host . $request_uri;
            set_transient( 'wppb_2fa_redirect_' . $current_user->ID, $current_url, HOUR_IN_SECONDS );
        }

        wp_redirect( $redirect_url );
        exit;
    }

    /**
     * Generate backup codes for a user
     *
     * @param int $count Number of codes to generate (default 10)
     * @return array Array with 'plain' codes and 'hashed' codes
     */
    function generate_backup_codes( $count = 10 ) {
        $plain_codes = array();
        $hashed_codes = array();

        for ( $i = 0; $i < $count; $i++ ) {
            // Generate 16-digit numeric code
            $code = '';
            for ( $j = 0; $j < 16; $j++ ) {
                $code .= wp_rand( 0, 9 );
            }

            $plain_codes[] = $code;
            $hashed_codes[] = array(
                'code_hash' => wp_hash_password( $code ),
                'used' => false
            );
        }

        return array(
            'plain' => $plain_codes,
            'hashed' => $hashed_codes
        );
    }

    /**
     * Save backup codes for a user
     *
     * @param int $user_id User ID
     * @param array $hashed_codes Array of hashed codes
     * @return bool True on success, false on failure
     */
    function save_backup_codes( $user_id, $hashed_codes ) {
        $result = update_user_option( $user_id, 'wppb_auth_backup_codes', $hashed_codes, true );
        if ( $result ) {
            update_user_option( $user_id, 'wppb_auth_backup_codes_generated', time(), true );
        }
        return $result;
    }

    /**
     * Verify a backup code for a user
     *
     * @param int $user_id User ID
     * @param string $input_code The code to verify
     * @return bool True if valid and unused, false otherwise
     */
    function verify_backup_code( $user_id, $input_code ) {
        $backup_codes = get_user_option( 'wppb_auth_backup_codes', $user_id );

        if ( !is_array( $backup_codes ) ) {
            return false;
        }

        foreach ( $backup_codes as $key => $code_data ) {
            if ( !$code_data['used'] && wp_check_password( $input_code, $code_data['code_hash'] ) ) {
                // Mark code as used
                $backup_codes[$key]['used'] = true;
                update_user_option( $user_id, 'wppb_auth_backup_codes', $backup_codes, true );
                return true;
            }
        }

        return false;
    }

    /**
     * Get count of remaining unused backup codes for a user
     *
     * @param int $user_id User ID
     * @return int Number of unused codes
     */
    function get_remaining_backup_codes_count( $user_id ) {
        $backup_codes = get_user_option( 'wppb_auth_backup_codes', $user_id );

        if ( !is_array( $backup_codes ) ) {
            return 0;
        }

        $unused_count = 0;
        foreach ( $backup_codes as $code_data ) {
            if ( !$code_data['used'] ) {
                $unused_count++;
            }
        }

        return $unused_count;
    }

    /**
     * Regenerate backup codes for a user
     *
     * @param int $user_id User ID
     * @return array Array with 'plain' codes and 'hashed' codes
     */
    function regenerate_backup_codes( $user_id ) {
        $codes = $this->generate_backup_codes( 10 );
        $this->save_backup_codes( $user_id, $codes['hashed'] );
        return $codes;
    }

    /**
     * Check if user can regenerate backup codes (rate limiting)
     *
     * @param int $user_id User ID
     * @return bool True if can regenerate, false if rate limited
     */
    function can_regenerate_backup_codes( $user_id ) {
        $last_generated = get_user_option( 'wppb_auth_backup_codes_generated', $user_id );
        if ( !$last_generated ) {
            return true;
        }

        // Rate limit: 60 seconds between regenerations
        return ( time() - $last_generated ) > 60;
    }

    /**
     * AJAX handler for regenerating backup codes
     */
    function ajax_regenerate_backup_codes() {
        // Security check
        check_ajax_referer( 'WPPBAuthaction', 'nonce' );

        $user_id = get_current_user_id();
        if ( !$user_id ) {
            wp_send_json_error( __( 'User not logged in.', 'profile-builder' ) );
        }

        // Check rate limiting
        if ( !$this->can_regenerate_backup_codes( $user_id ) ) {
            wp_send_json_error( __( 'Please wait before regenerating backup codes again.', 'profile-builder' ) );
        }

        // Regenerate codes
        $codes = $this->regenerate_backup_codes( $user_id );

        wp_send_json_success( array(
            'codes' => $codes['plain'],
            'count' => count( $codes['plain'] ),
            'message' => __( 'Backup codes generated', 'profile-builder' )
        ) );
    }

    function add_2fa_class_to_login_form( $form_args ) {

        $classes = array();

        if ( isset( $_GET['login_auth'] ) && $_GET['login_auth'] === 'true' ){
            $classes[] = 'wppb-2fa-authentication-requested';
        }

        if( isset( $form_args['form_classes'] ) && !in_array( 'wppb-2fa-form', $form_args['form_classes'] ) ){
            $form_args['form_classes'] = array_merge( $form_args['form_classes'], $classes );
        } else if( !isset( $form_args['form_classes'] ) ){
            $form_args['form_classes'] = array_merge( array( 'wppb-2fa-form' ), $classes );
        }

        return $form_args;

    }
}